<?php
// IngressoPratico/VenueBundle/Twig/SeatMapExtension.php

namespace IngressoPratico\VenueBundle\Twig;

use IngressoPratico\VenueBundle\Entity\SeatMap;
use IngressoPratico\VenueBundle\Entity\Plottable;

class SeatMapExtension extends \Twig_Extension
{
    /**
     * @var array
     */
    protected $templates;

    /**
     * @var Twig_Environment
     */
    protected $environment;

    /**
     * @var array
     */
    protected $seatTypeCss = array();

    /**
     * Constructor.
     *
     * @param array|string $templates  The Twig templates to use
     */
    public function __construct($templates)
    {
        if (is_string($templates)) {
            $templates = array($templates);
        }

        $this->templates = $templates;
    }

    /**
     * @{inheritDoc}
     */
    public function initRuntime(\Twig_Environment $environment)
    {
        $this->environment = $environment;
    }

    /**
     * @{inheritDoc}
     */
    public function getFunctions()
    {
        return array(
            'seatmap_render'      => new \Twig_Function_Method($this, 'renderSeatMap', array('is_safe' => array('html'))),
            'seatmap_render_item' => new \Twig_Function_Method($this, 'renderSeatMapItem', array('is_safe' => array('html'))),
        );
    }

    /**
     * @{inheritDoc}
     */
    public function getName()
    {
        return 'seatmap';
    }

    /**
     * Returns the token parser instance to add to the existing list.
     *
     * @return array An array of Twig_TokenParser instances
     */
    public function getTokenParsers()
    {
        return array(
            // {% seatmap_theme "SomeBundle::widgets.twig" %}
            new SeatMapThemeTokenParser(),
        );
    }

    /**
     * Add a template to look for seat map's blocks.
     *
     * @param string $template
     */
    public function addTemplate($template)
    {
        $this->templates[] = $template;
    }

    /**
     * Renders the seat map HTML representation. In your Twig template:
     *
     * {{ seatmap_render(seatmap) }}
     * {{ seatmap_render(seatmap, { 'image_dir': 'path/to/images' }) }}
     *
     * @param  SeatMap      $seatMap
     * @param  array        $variables
     * @return string|null
     */
    public function renderSeatMap(SeatMap $seatMap, array $variables = array())
    {
        return $this->render('seatmap', array(
            'items'     => $seatMap->getItems(),
            'variables' => $variables,
        ));
    }

    /**
     * Renders a plottable item HTML representation. In your Twig template:
     *
     * {{ seatmap_render_item(item) }}
     * {{ seatmap_render_item(item, { 'image_dir': 'path/to/images' }) }}
     *
     * @param  Plottable  $item
     * @param  array      $variables
     * @return string|null
     */
    public function renderSeatMapItem(Plottable $item, array $variables = array())
    {
        $variables = array_merge(
            array('item' => $item, 'image_dir' => null),
            $variables
        );

        return $this->render($item->getTypeName(), $variables);
    }

    /**
     * Renders a template block, passing the variables to the template.
     *
     * @param  string       $section
     * @param  array        $variables
     * @return string|null
     */
    private function render($section, array $variables)
    {
        if ($section != 'seatmap') {
            $section = 'seatmap_'.$section;
        }

        $templates = $this->getTemplates();

        if (isset($templates[$section])) {
            return $templates[$section]->renderBlock($section, $variables);
        }

        return null;
    }

    /**
     * Returns the availables blocks in the chosen template. You can override the default seatmap blocks using the
     * seatmap_theme tag in your Twig template:
     *
     * {% seatmap_theme _self 'SomeBundle::seatmap-blocks.html.twig' %}
     *
     * @return array
     */
    private function getTemplates()
    {
        $templates = array();

        foreach ($this->templates as &$template)
        {
            // Load template if not already loaded
            if (!$template instanceof \Twig_Template) {
                $template = $this->environment->loadTemplate($template);
            }

            $blocks = array();
            foreach ($template->getBlockNames() as $name) {
                $blocks[$name] = $template;
            }

            $templates = array_replace($templates, $blocks);
        }

        return $templates;
    }
}


// IngressoPratico/VenueBundle/Twig/SeatMapThemeNode.php

namespace IngressoPratico\VenueBundle\Twig;

class SeatMapThemeNode extends \Twig_Node
{
    public function __construct(\Twig_NodeInterface $resources, $lineno, $tag = null)
    {
        parent::__construct(array('resources' => $resources), array(), $lineno, $tag);
    }

    /**
     * Compiles the node to PHP.
     *
     * @param \Twig_Compiler $compiler A Twig_Compiler instance
     */
    public function compile(\Twig_Compiler $compiler)
    {
        $compiler->addDebugInfo($this);

        foreach ($this->getNode('resources') as $resource) {
            $compiler
                ->write('echo $this->env->getExtension(\'seatmap\')->addTemplate(')
                ->subcompile($resource)
                ->raw(');')
            ;
        }
    }
}


// IngressoPratico/VenueBundle/Twig/SeatMapThemeTokenParser.php

namespace IngressoPratico\VenueBundle\Twig;

class SeatMapThemeTokenParser extends \Twig_TokenParser
{
    /**
     * Parses a token and returns a node.
     *
     * @param  \Twig_Token  $token A Twig_Token instance
     * @return \Twig_NodeInterface A Twig_NodeInterface instance
     */
    public function parse(\Twig_Token $token)
    {
        $lineno = $token->getLine();
        $stream = $this->parser->getStream();

        $resources = array();
        do {
            $resources[] = $this->parser->getExpressionParser()->parseExpression();
        } while (!$stream->test(\Twig_Token::BLOCK_END_TYPE));

        $stream->expect(\Twig_Token::BLOCK_END_TYPE);

        return new SeatMapThemeNode(new \Twig_Node($resources), $lineno, $this->getTag());
    }

    /**
     * Gets the tag name associated with this token parser.
     *
     * @return string The tag name
     */
    public function getTag()
    {
        return 'seatmap_theme';
    }
}